<?php
/**
* The Editor API.
*
* @author Imran Khan <imran.khan@interspire.com>
*
* @package API
* @subpackage Editors_API
*/

/**
* Include the base API class if we haven't already.
*/
require_once(dirname(__FILE__) . '/api.php');

/**
* This will load the Editor, save an Editors, set details and get details.
* It will also check access areas and associate the editors.
*
* @package API
* @subpackage Editors_API
*/
class Editors_API extends API
{

	/**
	* Name of the Editor that is loaded.
	*
	* @var Int
	*/
	public $name = '';

	/**
	* Description of the Editor that we've loaded.
	*
	* @var String
	*/
	public $description = '';

	/**
	* Status of the Editor that we've loaded.
	*
	* @var Int
	*/
	public $status = 1;

	/**
	* The default value of the Editor
	*
	* @var Int
	*/
	public $default = 0;


	/**
	* Constructor
	* Sets up the database object, loads the autoresponder if the ID passed in is not 0.
	*
	* @param Int $autoresponderid The autoresponderid of the autoresponder to load. If it is 0 then you get a base class only. Passing in a autoresponderid > 0 will load that autoresponder.
	*
	* @see GetDb
	* @see Load
	*
	* @return True Always returns true
	*/
	function __construct()
	{
		$this->GetDb();

		return true;
	}

	/**
	* Load
	* Loads up the autoresponder and sets the appropriate class variables.
	*
	* @param Int $autoresponderid The autoresponderid to load up. If the autoresponderid is not present then it will not load up.
	*
	* @return Bool Will return false if the autoresponderid is not present, or the autoresponder can't be found, otherwise it set the class vars and return true.
	*/
	function Load($editorid=0)
	{
		$editorid = (int)$editorid;
		if ($editorid <= 0) {
			return false;
		}

		$query = 'SELECT * FROM ' . SENDSTUDIO_TABLEPREFIX . 'editors WHERE id=\'' . $editorid . '\'';
		$result = $this->Db->Query($query);
		if (!$result) {
			return false;
		}

		$editor = $this->Db->Fetch($result);
		if (empty($editor)) {
			return false;
		}

		$this->name = $editor['name'];
		$this->description = $editor['description'];
		$this->default =  $editor['default'];
		$this->status = $editor['status'];

		return true;
	}



	/**
	* GetEditors
	* Get a list of active editors.
	*
	* @param Bool $userSetting (TRUE | False)
	*
	* @return Mixed|FALSE Returns Editor record if successful, FALSE otherwise
	*/
	function GetEditors($userSetting=false)
	{    
	    $currentUser = IEM::userGetCurrent();
		$usewysiwyg = (isset($_GET['UserID']) ? GetUser($_GET['UserID'])->usewysiwyg : 1);
		
		// Check if Unlayer is enabled system-wide
		$settings_api = new Settings_API();
		$system_unlayer_enabled = (bool)$settings_api->Get('ENABLE_UNLAYER_SYSTEM');
		
		if($userSetting) {
			$query = 'SELECT * FROM ' . SENDSTUDIO_TABLEPREFIX . 'editors WHERE status=1';
			$result = $this->Db->Query($query);
			if (!$result) {
				list($error, $level) = $this->Db->GetError();
				trigger_error($error, $level);
				return false;
			}

			$GLOBALS['EditorList'] = '<option value="">Select template</option>';
			while ($row = $this->Db->Fetch($result)) {
				// Skip Unlayer if system setting is disabled
				if (!$system_unlayer_enabled && strtolower($row['name']) == 'unlayer') {
					continue;
				}
				$selected = ($usewysiwyg == $row['id']) ? ' selected' : '';
				$GLOBALS['EditorList'] .= '<option value="'.$row['id'].'" '.$selected.'>'.$row['description'].'</option>';
			}
		} else {
			$query = 'SELECT * FROM ' . SENDSTUDIO_TABLEPREFIX . 'editors WHERE status=1';
			$result = $this->Db->Query($query);
			if (!$result) {
				list($error, $level) = $this->Db->GetError();
				trigger_error($error, $level);
				return false;
			}

			$GLOBALS['EditorList'] = '';
			while ($row = $this->Db->Fetch($result)) {
				// Skip Unlayer if system setting is disabled
				if (!$system_unlayer_enabled && strtolower($row['name']) == 'unlayer') {
					continue;
				}
				$selected = ($currentUser->usewysiwyg == $row['id']) ? ' selected' : '';
				$GLOBALS['EditorList'] .= '<option value="'.$row['id'].'" '.$selected.'>'.$row['description'].'</option>';
			}
		}
		return $GLOBALS['EditorList'];
	}

	/**
	 * GetRecordByEditorID
	 * Get Editor by Editor ID
	 *
	 * @param Int $editorid Editor ID
	 * @return Array|FALSE Returns Editor record if successful, FALSE otherwise
	 */
	function GetRecordByEditorID($editorid)
	{

		$query = 'SELECT * FROM ' . SENDSTUDIO_TABLEPREFIX . 'editors WHERE id=\'' . $editorid . '\'';
		$result = $this->Db->Query($query);
		if (!$result) {
			list($error, $level) = $this->Db->GetError();
			trigger_error($error, $level);
			return false;
		}

		$row = $this->Db->Fetch($result);


		if (empty($row)) {
			return [];
		}
		$GLOBALS['EditorList'] = '<option value="' . $row['id'] . '" selected>' . $row['description'] . '</option>';


		return  $GLOBALS['EditorList'];
	}

	/**
	 * GetRecordByEditorName
	 * Get Editor by Editor ID
	 *
	 * @param Int $editorid Editor ID
	 * @return Array|FALSE Returns Editor record if successful, FALSE otherwise
	 */
	function GetRecordByEditorName($editorid)
	{
		 if (!$this->_checkTableExists()) {
			return false;
		}
		$query = 'SELECT * FROM ' . SENDSTUDIO_TABLEPREFIX . 'editors WHERE id=\'' . $editorid . '\'';
		$result = $this->Db->Query($query);
		if (!$result) {
			list($error, $level) = $this->Db->GetError();
			trigger_error($error, $level);
			return false;
		}

		$row = $this->Db->Fetch($result);

		if (empty($row)) {
			return [];
		}

		return   $row;
	}
	/**
	 * _checkTableExists
	 * Check whether or not the editors table exists
	 *
	 * @return Integer|FALSE Returns 1 if table exists, 0 if it doesn't, FALSE if it encountered any error
	 */
	protected function _checkTableExists()
	{
		if (SENDSTUDIO_DATABASE_TYPE == 'mysql') {
			$query = "SHOW TABLES LIKE '[|PREFIX|]editors'";
		} else {
			$query = "SELECT table_name FROM information_schema.tables WHERE table_name='[|PREFIX|]editors'";
		}

		$result = $this->Db->Query($query);
		if (!$result) {
			list($msg, $errno) = $this->Db->GetError();
			trigger_error($msg, $errno);
			return false;
		}

		$row = $this->Db->Fetch($result);
		$this->Db->FreeResult($result);

		if (empty($row)) {
			return 0;
		}

		return 1;
	}
}
